bl_info = {
    "name": "MatchSmile Studio",
    "author": "Dr. Amr Touny",
    "version": (1, 1, 0),
    "blender": (3, 0, 0),
    "location": "View3D > N-Panel > MatchSmile",
    "description": "Face & Expression Manager with IO Tools",
    "category": "Animation",
}

import bpy
import bpy.utils.previews
import os
import base64

# --- EMBEDDED LOGO DATA ---
# Icon is now loaded from icon.png in the package

# Store previews globally

# Store previews globally
preview_collections = {}

# 52 ARKit Blendshapes 
ARKIT_SHAPES = [
    "browDownLeft", "browDownRight", "browInnerUp", "browOuterUpLeft", "browOuterUpRight",
    "cheekPuff", "cheekSquintLeft", "cheekSquintRight", "eyeBlinkLeft", "eyeBlinkRight",
    "eyeLookDownLeft", "eyeLookDownRight", "eyeLookInLeft", "eyeLookInRight", "eyeLookOutLeft",
    "eyeLookOutRight", "eyeLookUpLeft", "eyeLookUpRight", "eyeSquintLeft", "eyeSquintRight",
    "eyeWideLeft", "eyeWideRight", "jawForward", "jawLeft", "jawOpen", "jawRight",
    "mouthClose", "mouthDimpleLeft", "mouthDimpleRight", "mouthFrownLeft", "mouthFrownRight",
    "mouthFunnel", "mouthLeft", "mouthLowerDownLeft", "mouthLowerDownRight", "mouthPressLeft",
    "mouthPressRight", "mouthPucker", "mouthRight", "mouthRollLower", "mouthRollUpper",
    "mouthShrugLower", "mouthShrugUpper", "mouthSmileLeft", "mouthSmileRight", "mouthStretchLeft",
    "mouthStretchRight", "mouthUpperUpLeft", "mouthUpperUpRight", "noseSneerLeft", "noseSneerRight",
    "tongueOut"
]

# ========================================
# 1. IO OPERATORS
# ========================================

class MATCHSMILE_OT_ImportOBJ(bpy.types.Operator):
    """Import Static Mesh (OBJ)"""
    bl_idname = "matchsmile.import_obj"
    bl_label = "Import OBJ"
    
    def execute(self, context):
        if bpy.app.version >= (4, 0, 0):
             bpy.ops.wm.obj_import('INVOKE_DEFAULT')
        else:
             bpy.ops.import_scene.obj('INVOKE_DEFAULT')
        return {'FINISHED'}

class MATCHSMILE_OT_ImportGLB(bpy.types.Operator):
    """Import Animated Mesh (GLB/GLTF)"""
    bl_idname = "matchsmile.import_glb"
    bl_label = "Import GLB"
    
    def execute(self, context):
        bpy.ops.import_scene.gltf('INVOKE_DEFAULT')
        return {'FINISHED'}

class MATCHSMILE_OT_ImportJSON(bpy.types.Operator):
    """Import Motion Capture JSON"""
    bl_idname = "matchsmile.import_json"
    bl_label = "Import Motion JSON"
    filepath: bpy.props.StringProperty(subtype="FILE_PATH")
    filter_glob: bpy.props.StringProperty(default="*.json", options={'HIDDEN'})

    def invoke(self, context, event):
        context.window_manager.fileselect_add(self)
        return {'RUNNING_MODAL'}

    def execute(self, context):
        import json
        obj = context.object
        
        # Validation
        if not obj:
            self.report({'ERROR'}, "No active object selected")
            return {'CANCELLED'}
        
        # Auto-Create Validation Shape Keys if completely missing
        if not obj.data.shape_keys:
             basis = obj.shape_key_add(name="Basis")
             basis.interpolation = 'KEY_LINEAR'
             
        key_blocks = obj.data.shape_keys.key_blocks
        
        # Ensure ARKit shapes exist
        for shape_name in ARKIT_SHAPES:
            if shape_name not in key_blocks:
                 new_shape = obj.shape_key_add(name=shape_name)
                 new_shape.value = 0.0

        try:
            with open(self.filepath, 'r') as f:
                data = json.load(f)
            
            frames = data.get("frames", [])
            fps = data.get("fps", 30)
            
            context.scene.render.fps = int(fps)
            context.scene.frame_start = 1
            context.scene.frame_end = len(frames)
            
            print(f"Importing {len(frames)} frames at {fps} FPS...")
            
            for i, frame_data in enumerate(frames):
                frame_num = 1 + i
                shapes = frame_data.get("shapes", [])
                
                for shape in shapes:
                    name = shape.get("categoryName")
                    val = shape.get("score", 0.0)
                    
                    if name in key_blocks:
                        key_blocks[name].value = val
                        key_blocks[name].keyframe_insert("value", frame=frame_num)
                        
            self.report({'INFO'}, f"Motion imported: {len(frames)} frames")
            return {'FINISHED'}
            
        except Exception as e:
            self.report({'ERROR'}, f"Import Failed: {str(e)}")
            return {'CANCELLED'}

class MATCHSMILE_OT_AutoRetarget(bpy.types.Operator):
    """Automated Retargeting: Loads Canonical Face & Conforms to Scan"""
    bl_idname = "matchsmile.auto_retarget"
    bl_label = "Auto-Retarget Canonical Face"
    
    def execute(self, context):
        scan = context.object
        if not scan or scan.type != 'MESH':
            self.report({'ERROR'}, "Please select your imported Face Scan first")
            return {'CANCELLED'}
            
        # 1. Locate Bundled Canonical Face
        script_file = os.path.realpath(__file__)
        addon_dir = os.path.dirname(script_file)
        canonical_path = os.path.join(addon_dir, "canonical_face_model.fbx")
        
        if not os.path.exists(canonical_path):
            self.report({'ERROR'}, "Canonical Face Model not found in addon folder")
            return {'CANCELLED'}
            
        # 2. Import Canonical Face
        bpy.ops.import_scene.fbx(filepath=canonical_path)
        rig = context.object # The imported FBX becomes active
        rig.name = "MatchSmile_Rigged_Face"
        
        # 3. Setup Shrinkwrap (Conform)
        sw = rig.modifiers.new(name="MatchSmile_Conform", type='SHRINKWRAP')
        sw.target = scan
        sw.wrap_method = 'PROJECT'
        sw.use_negative_direction = True
        sw.use_positive_direction = True
        sw.offset = 0.0005 # Tiny offset
        
        # 4. Setup Corrective Smooth
        sm = rig.modifiers.new(name="MatchSmile_Smooth", type='CORRECTIVE_SMOOTH')
        sm.factor = 0.6
        sm.iterations = 10
        sm.smooth_type = 'SIMPLE'
        
        # 5. Transfer Material (Texture)
        if len(scan.data.materials) > 0:
            rig.data.materials.clear()
            rig.data.materials.append(scan.data.materials[0])
            
        # 6. Hide Original Scan
        scan.hide_viewport = True
        scan.hide_render = True
        
        # 7. Select New Rig
        bpy.ops.object.select_all(action='DESELECT')
        rig.select_set(True)
        context.view_layer.objects.active = rig
        
        self.report({'INFO'}, "Retargeting Complete! Ready for Motion JSON.")
        return {'FINISHED'}

class MATCHSMILE_OT_ExportFBX(bpy.types.Operator):
    """Export Scene to FBX"""
    bl_idname = "matchsmile.export_fbx"
    bl_label = "Export FBX"
    
    def execute(self, context):
        bpy.ops.export_scene.fbx('INVOKE_DEFAULT', use_selection=True)
        return {'FINISHED'}

# ========================================
# 2. UI PANELS
# ========================================

class MATCHSMILE_PT_MainPanel(bpy.types.Panel):
    """Main Panel with Logo and IO Tools"""
    bl_label = "MatchSmile Studio"
    bl_idname = "MATCHSMILE_PT_MainPanel"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "MatchSmile"

    def draw(self, context):
        layout = self.layout
        pcoll = preview_collections.get("main")
        my_icon = pcoll.get("logo_v2") if pcoll else None

        # Header with Logo
        row = layout.row()
        if my_icon:
            row.label(text="MatchSmile Engine", icon_value=my_icon.icon_id)
        else:
            row.label(text="MatchSmile Engine", icon='IMPORT')

        layout.separator()
        
        # IO Tools Sub-section
        box = layout.box()
        box.label(text="IO Tools", icon='FILE_FOLDER')
        
        col = box.column(align=True)
        col.operator("matchsmile.import_obj", icon='IMPORT')
        col.operator("matchsmile.import_glb", icon='SCENE_DATA')
        col.operator("matchsmile.import_json", icon='FILE_SCRIPT')
        col.separator()
        col.label(text="Retargeting", icon='MOD_SHRINKWRAP')
        col.operator("matchsmile.auto_retarget", icon='ARMATURE_DATA')
        col.separator()
        col.operator("matchsmile.export_fbx", icon='EXPORT')

class MATCHSMILE_PT_Expressions(bpy.types.Panel):
    """Reallusion-Style Expression Manager"""
    bl_label = "Face Expressions"
    bl_idname = "MATCHSMILE_PT_Expressions"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "MatchSmile"
    bl_parent_id = "MATCHSMILE_PT_MainPanel"
    bl_options = {'DEFAULT_CLOSED'}

    def draw(self, context):
        layout = self.layout
        obj = context.object

        # Check if active object is a Mesh with Shape Keys
        if not obj or obj.type != 'MESH':
            layout.label(text="Select a Mesh Object", icon='INFO')
            return
        
        if not obj.data.shape_keys:
            layout.label(text="No Shape Keys Found", icon='ERROR')
            return

        key_blocks = obj.data.shape_keys.key_blocks
        
        box = layout.box()
        col = box.column()

        # Iterate through Standard 52 Shapes
        found_count = 0
        for shape_name in ARKIT_SHAPES:
            row = col.row(align=True)
            
            if shape_name in key_blocks:
                # Active Slider
                row.prop(key_blocks[shape_name], "value", text=shape_name, slider=True)
                found_count += 1
            else:
                # Missing Indicator
                row.enabled = False
                row.label(text=f"Missing: {shape_name}", icon='DOT')
        
        if found_count == 0:
            layout.label(text="No ARKit Shapes Detected", icon='INFO')

# ========================================
# 3. REGISTRATION
# ========================================

classes = (
    MATCHSMILE_OT_ImportOBJ,
    MATCHSMILE_OT_ImportGLB,
    MATCHSMILE_OT_ImportJSON,
    MATCHSMILE_OT_AutoRetarget,
    MATCHSMILE_OT_ExportFBX,
    MATCHSMILE_PT_MainPanel,
    MATCHSMILE_PT_Expressions,
)

def register():
    # 1. Register Classes
    for cls in classes:
        bpy.utils.register_class(cls)

    # 2. Register Custom Icon
    pcoll = bpy.utils.previews.new()
    
    # Load icon directly from addon folder
    icons_dir = os.path.join(os.path.dirname(__file__))
    icon_path = os.path.join(icons_dir, "icon.png")
    
    if os.path.exists(icon_path):
        pcoll.load("logo_v2", icon_path, 'IMAGE')
    else:
        print("MatchSmile Studio: Icon not found at", icon_path)
        
    preview_collections["main"] = pcoll

def unregister():
    # 1. Unregister Classes
    for cls in reversed(classes):
        bpy.utils.unregister_class(cls)

    # 2. Unregister Icon
    for pcoll in preview_collections.values():
        bpy.utils.previews.remove(pcoll)
    preview_collections.clear()

if __name__ == "__main__":
    register()
